/*
 *    Simple LED control API implemenation for TriBoard-TC2xxx and AppKit-TC2xx
 */

#include "ledctl.h"

#include "tc_inc_path.h"

#include TC_INCLUDE(TCPATH/IfxGtm_reg.h)
#include TC_INCLUDE(TCPATH/IfxGtm_bf.h)
#include TC_INCLUDE(TCPATH/IfxPort_reg.h)

#if (RUN_ON_APPKIT > 0)
#if (APPKIT_TC2X7 > 0)
/* AppKit-TC2xx: P13.0 .. P13.3 --> LED D107 ... D110 */

#define LEDCTL_LED_COUNT	4
#define LEDCTL_LED_FIRST	0
#else
/* AppKit-TC2xx: P33.8 .. P33.11 --> LED D107 ... D110 */

#define LEDCTL_LED_COUNT	4
#define LEDCTL_LED_FIRST	8
#endif /* APPKIT_TC2X7 */
#else
/* TriBoard-TC2xxx : P33.6 ... P33.13 --> LED D501 ... D508 */

#define LEDCTL_LED_COUNT	8
#define LEDCTL_LED_FIRST	6
#endif /* RUN_ON_APPKIT > 0 */

#define MASK_ALL_LEDS		((1 << LEDCTL_LED_COUNT) - 1)

#if (APPKIT_TC2X7 > 0)
static Ifx_P * const portLED = (Ifx_P *)&MODULE_P13;
#if defined(__DEBUG_WITH_SCOPE__)
static Ifx_P * const portIOC = (Ifx_P *)&MODULE_P02;
#endif
#else
static Ifx_P * const portLED = (Ifx_P *)&MODULE_P33;
#endif /* APPKIT_TC2X7 */


int LEDCTL_Init(void)
{
#if (RUN_ON_APPKIT > 0)
#if (APPKIT_TC2X7 > 0)
	/* initialise all LEDs (P13.0 .. P13.3) */
  portLED->IOCR0.U = 0x80808080;  /* OUT_PPGPIO */
#if defined(__DEBUG_WITH_SCOPE__)
  portIOC->IOCR0.U = 0x80808080;  /* OUT_PPGPIO */
#endif
#else
	/* initialise all LEDs (P33.8 .. P33.11) */
	portLED->IOCR8.U = 0x80808080;	/* OUT_PPGPIO */
#endif /* APPKIT_TC2X7 */
#else
	/* initialise all LEDs (P33.6 .. P33.13) */
	portLED->IOCR4.B.PC6 = 0x10;	/* OUT_PPGPIO */
	portLED->IOCR4.B.PC7 = 0x10;	/* OUT_PPGPIO */
	portLED->IOCR8.U = 0x80808080;	/* OUT_PPGPIO */
	portLED->IOCR12.B.PC12 = 0x10;	/* OUT_PPGPIO */
	portLED->IOCR12.B.PC13 = 0x10;	/* OUT_PPGPIO */
#endif /* RUN_ON_APPKIT > 0 */
	/* all LEDs OFF */
	portLED->OMR.U = (MASK_ALL_LEDS << LEDCTL_LED_FIRST);
#if defined(__DEBUG_WITH_SCOPE__)
  portIOC->OMR.U = (MASK_ALL_LEDS << LEDCTL_LED_FIRST);
#endif
	return LEDCTL_LED_COUNT;
}

void LEDCTL_Cleanup(void)
{
}

int LEDCTL_GetCount(void)
{
	return LEDCTL_LED_COUNT;
}

void LEDCTL_On(int Index)
{
	if (Index < LEDCTL_LED_COUNT)
	{
		/* set PCLx pin */
		portLED->OMR.U = (1 << (Index + LEDCTL_LED_FIRST + 16));
#if defined(__DEBUG_WITH_SCOPE__)
    portIOC->OMR.U = (1 << (Index + LEDCTL_LED_FIRST + 16));
#endif
	}
}

void LEDCTL_Off(int Index)
{
	if (Index < LEDCTL_LED_COUNT)
	{
		/* set PSx pin */
		portLED->OMR.U = (1 << (Index + LEDCTL_LED_FIRST));
#if defined(__DEBUG_WITH_SCOPE__)
    portIOC->OMR.U = (1 << (Index + LEDCTL_LED_FIRST));
#endif
	}
}

int LEDCTL_Status(int Index)
{
	if (Index < LEDCTL_LED_COUNT)
	{
		unsigned int mask = (1 << (Index + LEDCTL_LED_FIRST));
		/* LEDs are low active */
		return ((portLED->OUT.U & mask) ? 0 : 1);
	}
	else
	{
		return -1;
	}
}

void LEDCTL_Toggle(int Index)
{
	if (Index < LEDCTL_LED_COUNT)
	{
		/* set PCLx and PSx pin to 1 ==> toggle pin state */
		portLED->OMR.U = (1 << (Index + LEDCTL_LED_FIRST + 16)) | (1 << (Index + LEDCTL_LED_FIRST));
#if defined(__DEBUG_WITH_SCOPE__)
    portIOC->OMR.U = (1 << (Index + LEDCTL_LED_FIRST + 16)) | (1 << (Index + LEDCTL_LED_FIRST));
#endif
	}
}

void LEDCTL_Out(unsigned int uiValue)
{
	unsigned int actVal = portLED->OUT.U;
	actVal |= (MASK_ALL_LEDS << LEDCTL_LED_FIRST);
	actVal ^= ((uiValue & MASK_ALL_LEDS) << LEDCTL_LED_FIRST);
	portLED->OUT.U = actVal;
#if defined(__DEBUG_WITH_SCOPE__)
  portIOC->OUT.U = actVal;
#endif
}
